% Razporeditev tock glede na fiksne tocke
%
% fiksT = [x1 y1;
%          x2 y2;
%           ... 
%          xN yN]
%
% ter na podane razdalje 
%
% R = [i1, j1, d_i1j1;
%      i2, j2, d_i2j2;
%          ...
%      iM, jM, d_iMjM],
%
% kjer indeksa i in j (i, j >= 1) oznacujeta, za kateri tocki gre. Indeksi,
% ki ne presegajo N, predtavljajo tocko fiksT(indeks, :). Smiselno je, da za
% vsako vrstico matrike R velja max(i, j) > N, saj v nasprotnem primeru enacbo
% ignoriramo, ker fiksnih tock ne premikamo.
%
% Z matriko R je podan predolocen sistem, ki ga resimo z Gauss-Newtonovo
% iteracijo za funkcijo F, kot je definirana v opisu metode FInJF spodaj.
%
% Ce tretjega argumenta ne podamo, bomo zacetni priblizek zgenerirali nakljucno.
% Ce ga, mora biti stolpec oblike
%
% [x(N + 1); y(N + 1); x(N + 2); y(N + 2); ... ].
function [koordinate, napaka] = razporejanjeTock(fiksT, R, zacetni_priblizek)

stFiksTock = length(fiksT(:,1));
stVsehTock = max(max(R(:,1:2)));

if nargin == 2
    stIskanihTock = stVsehTock - stFiksTock;
    koordinate = rand(2 * stIskanihTock, 1);
else
    koordinate = zacetni_priblizek;
end

toleranca = 10^-8;
maxIteracij = 100;
k = 1;
while true
    [F, JF] = FInJF(fiksT, R, koordinate, stFiksTock);
    popravek = -pinv(JF) * F;  % tu lahko popravimo tudi na JF \ F
    premik = norm(popravek);
    koordinate = koordinate + popravek;
    
    if k > maxIteracij
        fprintf('Dosegli smo dovoljeno stevilo iteracij %d.\n', maxIteracij);
        break;
    elseif premik < toleranca
        fprintf('Konvergenca po %d korakih.\n', k);
        break;
    end
    k = k + 1;
end

napaka = norm(F);

end


% Izracunamo F in njeno Jaccobijevo matriko v tocki trenutni_priblizek.
% F je oblike
%
% F(T(N + 1)_x, T(N + 1)_y, T(N + 2)_x, T(N + 2)_y, ...) = (F_ij(Ti, Tj))_ij,
%
% kjer gresta i in j po parih indeksov (i, j) iz vrstic matrike razdalj.
% Funkcija F_ij je definirana kot
%
% F_ij: Ti_x, Ti_y, Tj_x, Tj_y --> (Ti_x - Tj_x)^2 + (Ti_y - Tj_y)^2 - d_ij^2,
%
% kjer je d_ij tretje stevilo v tisti vrstici matrike R, ki pripada paru (i, j).
% 
% Opombi.
% 1) V resnici imajo funkcije F_ij enake argumente kot F. To upostevamo
%    pri gradnji matrike J, v zgornjem opisu pa smo zaradi preprostosti pri F_ij
%    napisali le pomembne stiri.
% 2) F_ij bi lahko imel tudi predpis
%    F_ij: Ti_x, Ti_y, Tj_x, Tj_y --> ||Ti - Tj||_2 - d_ij
%    Tako bi dobili iste 'nicle' (oz. lokalne minimume).
function [F, J] = FInJF(fiksne, matrika_razdalj, trenutni_priblizek, N)

J = zeros(size(matrika_razdalj, 1), size(trenutni_priblizek, 1));
F = zeros(size(matrika_razdalj, 1), 1);
for ind = 1:size(matrika_razdalj, 1)
    i = matrika_razdalj(ind, 1);
    j = matrika_razdalj(ind, 2);
    d = matrika_razdalj(ind, 3);

    [ind_xi, ind_yi, xi, yi] = pretvorba(i, N, fiksne, trenutni_priblizek);
    [ind_xj, ind_yj, xj, yj] = pretvorba(j, N, fiksne, trenutni_priblizek);
  
    dx = xi - xj;
    dy = yi - yj;
    % posodobimo F
    F(ind) = dx^2 + dy^2 - d^2;
    % posodobimo J
    if i > N
        J(ind, ind_xi) = 2 * dx;
        J(ind, ind_yi) = 2 * dy;
    end
    if j > N
        J(ind, ind_xj) = -2 * dx;
        J(ind, ind_yj) = -2 * dy;
    end
end

end


% Pomozna funkcija za FInJF. Izracunamo (morda trenutni) koordinati
% x in y tocke ter indeksa tocki pripadajocih stolpcev v Jaccobijevi matriki
% (ce gre za premicno tocko).
function [ind_x, ind_y, x, y] = pretvorba(indeks_tocke, N, fiksneT, premicne)
    if indeks_tocke > N
        ind_y = 2 * (indeks_tocke - N);
        ind_x = ind_y - 1;
        x = premicne(ind_x);
        y = premicne(ind_y);
    else
        ind_x = 0;  % ni vazno, kaj sta
        ind_y = 0;  % indeksa v tem primeru
        x = fiksneT(indeks_tocke, 1);
        y = fiksneT(indeks_tocke, 2);
    end
end