package si.lj.uni.fmf.pro2.lectures.networks;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.Stack;

public class Graph {
	
	private int m;
	
	private String name;
	
	private Map<Integer, String> labels;
	
	private Map<Integer, Integer> clusters;
	
	private Map<Integer, Set<Integer>> neighbors;
	
	public Graph(String name) {
		super();
		
		this.name = name;
		
		m = 0;
		labels = new HashMap<Integer, String>();
		clusters = new HashMap<Integer, Integer>();
		neighbors = new HashMap<Integer, Set<Integer>>();
	}
	
	public boolean isNode(int node) {
		return labels.containsKey(node);
	}
	
	public boolean addNode(int node) {
		return addNode(node, "" + node);
	}
	
	public boolean addNode(int node, String label) {
		return addNode(node, label, 1);
	}
	
	public boolean addNode(int node, int cluster) {
		return addNode(node, "" + node, cluster);
	}
	
	public boolean addNode(int node, String label, int cluster) {
		if (isNode(node))
			return false;

		labels.put(node, label);
		clusters.put(node, cluster);
		neighbors.put(node, new HashSet<Integer>());
		
		return true;
	}
	
	public boolean isEdge(Edge edge) {
		return isEdge(edge.getFirst(), edge.getSecond());
	}
	
	public boolean isEdge(int first, int second) {
		if (!isNode(first) || !isNode(second))
			return false;
		
		return neighbors.get(first).contains(second);
	}
	
	public boolean addEdge(Edge edge) {
		return addEdge(edge.getFirst(), edge.getSecond());
	}
	
	public boolean addEdge(int first, int second) {
		if (isEdge(first, second))
			return false;
		
		if (!isNode(first))
			throw new IllegalArgumentException("Node " + first + " does not exist");
		
		if (!isNode(second))
			throw new IllegalArgumentException("Node " + second + " does not exist");
		
		m++;
		neighbors.get(first).add(second);
		neighbors.get(second).add(first);
		
		return true;
	}
	
	public Set<Integer> getNodes() {
		return labels.keySet();
	}
	
	public Set<Edge> getEdges() {
		Set<Edge> edges = new HashSet<Edge>();
		for (int node: getNodes())
			for (int neighbor: getNeighbors(node))
				edges.add(new Edge(node, neighbor));
		return edges;
	}
	
	public String getLabel(int node) {
		if (!isNode(node))
			throw new IllegalArgumentException("Node " + node + " does not exist");
		
		return labels.get(node);
	}
	
	public Set<Integer> getNeighbors(int node) {
		if (!isNode(node))
			throw new IllegalArgumentException("Node " + node + " does not exist");
			
		return neighbors.get(node);
	}
	
	public int getDegree(int node) {
		return getNeighbors(node).size();
	}
	
	public int getCluster(int node) {
		if (!isNode(node))
			throw new IllegalArgumentException("Node " + node + " does not exist");
		
		return clusters.get(node);
	}
	
	public void setCluster(int node, int cluster) {
		if (!isNode(node))
			throw new IllegalArgumentException("Node " + node + " does not exist");
		
		clusters.put(node, cluster);
	}
	
	public String getName() {
		return name;
	}
	
	public int getN() {
		return labels.size();
	}
	
	public int getM() {
		return m;
	}
	
	public double getK() {
		return 2.0 * getM() / getN();
	}
	
	public int getD() {
		int d = 0;
		for (int node: getNodes())
			if (getDegree(node) > d)
				d = getDegree(node);
		
		return d;
	}
	
	public int getI() {
		int i = 0;
		for (int node: getNodes())
			if (getDegree(node) == 0)
				i++;
		
		return i;
	}
	
	public int getL() {
		int l = 0;
		for (int node: getNodes())
			for (int neighbor: getNeighbors(node))
				if (node == neighbor)
					l++;

		return l / 2;
	}
	
	public void clustering() {
		List<Integer> nodes = new ArrayList<Integer>(getNodes());
		
		Map<Integer, Integer> clustering = new HashMap<Integer, Integer>(); 	    		
		for (int node: nodes)
			clustering.put(node, node);

		boolean changed = true;
		while (changed) {
			changed = false;
			
			Collections.shuffle(nodes);
			
			for (int node: nodes) {
				Map<Integer, Integer> clusters = new HashMap<Integer, Integer>();
				clusters.put(clustering.get(node), 0);
				for (int neighbor: getNeighbors(node)) {
					if (!clusters.containsKey(clustering.get(neighbor)))
						clusters.put(clustering.get(neighbor), 0);
					clusters.put(clustering.get(neighbor), clusters.get(clustering.get(neighbor)) + 1);
				}
				
				if (clusters.size() > 0) {
					int cluster = clusters.keySet().iterator().next();
					for (int candidate: clusters.keySet())
						if (clusters.get(candidate) > clusters.get(cluster))
							cluster = candidate;
					
					if (clusters.get(cluster) > clusters.get(clustering.get(node))) {
						clustering.put(node, cluster);
						changed = true;
					}
				}
			}
		}
		
		Map<Integer, Integer> sizes = new HashMap<Integer, Integer>();
		for (int node: nodes) {
			if (!sizes.containsKey(clustering.get(node)))
				sizes.put(clustering.get(node), 0);
			sizes.put(clustering.get(node), sizes.get(clustering.get(node)) + 1);
		}
		
		List<Integer> clusters = new ArrayList<Integer>(sizes.keySet());
		Collections.sort(clusters, new Comparator<Integer>() {
			@Override
			public int compare(Integer first, Integer second) {
				return -sizes.get(first).compareTo(sizes.get(second));
			}
		});
		
		Map<Integer, Integer> mapping = new HashMap<Integer, Integer>();
		for (int i = 0; i < clusters.size(); i++)
			mapping.put(clusters.get(i), i + 1);
		
		for (int node: nodes)
			setCluster(node, mapping.get(clustering.get(node)));
	}
	
	@Override
	public String toString() {
		return String.format("%12s | %s\n%12s | %,d (%,d)\n%12s | %,d (%,d)\n%12s | %.4f\n", 
			"Graph", "'" + getName() + "'", "Nodes", getN(), getI(), "Edges", getM(), getL(), "Degree", getK());
	}

	public static Graph read(String file) throws IOException {
		return read(new File(file));
	}
	
	public static Graph read(File file) throws IOException {
		Graph graph = new Graph(file.getName().split("\\.")[0]);
		
		BufferedReader reader = new BufferedReader(new FileReader(file));
		
		String line = reader.readLine();
		while ((line = reader.readLine()) != null) {
			if (line.startsWith("*edges") || line.startsWith("*arcs"))
				break;
			
			String[] array = line.split("\"");
			if (array.length > 2)
				graph.addNode(Integer.parseInt(line.split(" ")[0]), array[1], Integer.parseInt(array[2].trim()));
			else if (array.length > 1)
				graph.addNode(Integer.parseInt(line.split(" ")[0]), array[1]);
			else
				graph.addNode(Integer.parseInt(line.split(" ")[0]));
		}
		
		while ((line = reader.readLine()) != null) {
			String[] nodes = line.split(" ");
			graph.addEdge(Integer.parseInt(nodes[0]), Integer.parseInt(nodes[1]));
		}
			
		reader.close();
		
		return graph;
	}
	
	public static void write(Graph graph, String file) throws IOException {
		write(graph, new File(file));
	}
	
	public static void write(Graph graph, File file) throws IOException {
		List<Integer> nodes = new ArrayList<Integer>(graph.getNodes());
		Collections.sort(nodes);
		
		BufferedWriter writer = new BufferedWriter(new FileWriter(file));
		
		writer.write("*vertices " + graph.getN() + "\n");
		for (int node: nodes)
			writer.write(node + " \"" + graph.getLabel(node) + "\" " + graph.getCluster(node) + "\n");
		
		writer.write("*edges " + graph.getM() + "\n");
		for (int node: nodes) {
			boolean loops = true;
			for (int neighbor: graph.getNeighbors(node))
				if (node < neighbor)
					writer.write(node + " " + neighbor + "\n");
				else if (node == neighbor) {
					if (loops)
						writer.write(node + " " + neighbor + "\n");
					loops = !loops;
				}
		}
		
		writer.flush();
		writer.close();
	}

	public static Graph random(int n, double k) {
		Graph graph = new Graph("Random");
		for (int i = 0; i < n; i++)
			graph.addNode(i + 1, (int)(3.0 * Math.random()));
		
		for (int i = 0; i < n; i++)
			for (int j = i + 1; j < n; j++)
				if (Math.random() < k / (n - 1))
					graph.addEdge(i + 1, j + 1);
		
		return graph;
	}
	
	public static List<Integer> components(Graph graph) {
		List<Integer> components = new ArrayList<Integer>();
		
		Set<Integer> nodes = new HashSet<Integer>(graph.getNodes());
		while (!nodes.isEmpty())
			components.add(component(graph, nodes));
		
		return components;
	}
	
	private static int component(Graph graph, Set<Integer> nodes) {
		int node = nodes.iterator().next();
		nodes.remove(node);
		
		Stack<Integer> stack = new Stack<Integer>();
		stack.push(node);
		
		int component = 0;
		while (!stack.isEmpty()) {
			for (int neighbor: graph.getNeighbors(stack.pop()))
				if (nodes.remove(neighbor))
					stack.push(neighbor);
			
			component++;
		}
		
		return component;
	}
	
}