package si.lj.uni.fmf.pro2.lectures.networks;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class Layouting {
	
	private Map<Integer, Point> layouting;
	
	public Layouting(Graph graph) {
		this(graph, true);
	}
	
	public Layouting(Graph graph, boolean random) {
		super();
		
		layouting = new HashMap<Integer, Point>();
		for (int node: graph.getNodes())
			layouting.put(node, random? new Point(): new Point(0.0, 0.0));
	}
	
	public void setPoint(int node, Point point) {
		if (!layouting.containsKey(node))
			throw new IllegalArgumentException("Node " + node + " does not exist");
		
		layouting.put(node, point);
	}
	
	public Point getPoint(int node) {
		if (!layouting.containsKey(node))
			throw new IllegalArgumentException("Node " + node + " does not exist");
		
		return layouting.get(node);
	}
	
	public Point get(int node) {
		return getPoint(node);
	}
	
	public double getX(int node) {
		return getPoint(node).getX();
	}
	
	public double getY(int node) {
		return getPoint(node).getY();
	}
	
	public static Layouting spring(Graph graph) {
		return new Spring(graph).layout();
	}
	
	public static Layouting circular(Graph graph) {
		List<Integer> nodes = new ArrayList<Integer>(graph.getNodes());
		Collections.shuffle(nodes);
		
		Layouting layouting = new Layouting(graph);
		for (int i = 0; i < nodes.size(); i++)
			layouting.setPoint(nodes.get(i), new Point(0.5 + 0.5 * Math.cos(2.0 * i * Math.PI / graph.getN()), 0.5 + 0.5 * Math.sin(2.0 * i * Math.PI / graph.getN())));
		
		return layouting;
	}
	
}

class Spring {
	
	protected Graph graph;
	
	protected Layouting layouting;

    private Layouting offsets;
    
    private double temperature;
    
    static final int ITERATIONS = 750;
    
    static final double STOCHASTIC = 0.2;
    
    static final double ATTRACTION = 0.75;
    
    static final double REPULSION = 0.5;
    
    static final double EPSILON = 1e-6;
    
    static final double SPAN = 0.85;
    
    public Spring(Graph graph) {
    	this.graph = graph;

    	layouting = new Layouting(graph, true);
    	offsets = new Layouting(graph, false);
    	temperature = 1.0;
    	
    	nodes = new ArrayList<Integer>(graph.getNodes());
    }
    
    public Layouting layout() {
    	int i = 0;
    	while (i <= ITERATIONS) {
    		offsets = new Layouting(graph, false);

        	for(int node: graph.getNodes())
        		repulsion(node);

        	for(Edge edge: graph.getEdges())
        		attraction(edge);

        	for(int node: graph.getNodes())
        		layout(node);

        	temperature *= (1 - 1.0 * i / ITERATIONS);
    		i++;
    	}
    	
    	double minX = Double.MAX_VALUE, maxX = -Double.MAX_VALUE;
    	double minY = Double.MAX_VALUE, maxY = -Double.MAX_VALUE;
    	for (int node: graph.getNodes()) {
    		if (layouting.get(node).getX() < minX)
    			minX = layouting.get(node).getX();
    		if (layouting.get(node).getX() > maxX)
    			maxX = layouting.get(node).getX();
    		if (layouting.get(node).getY() < minY)
    			minY = layouting.get(node).getY();
    		if (layouting.get(node).getY() > maxY)
    			maxY = layouting.get(node).getY();
    	}

    	if (maxX - minX < SPAN)
    		for (int node: graph.getNodes())
    			layouting.get(node).setX((layouting.get(node).getX() - minX) / (maxX - minX) * SPAN + (1 - SPAN) / 2);
    	if (maxY - minY < SPAN)
    		for (int node: graph.getNodes())
    			layouting.get(node).setY((layouting.get(node).getY() - minY) / (maxY - minY) * SPAN + (1 - SPAN) / 2);
    	
    	return layouting;
    }
    
    private void layout(Integer node) {
        double norm = Math.max(EPSILON, offsets.get(node).norm());

        layouting.get(node).setX(Math.max(0.0, Math.min(1.0, layouting.get(node).getX() + offsets.getX(node) / norm * Math.min(norm, temperature))));
        layouting.get(node).setY(Math.max(0.0, Math.min(1.0, layouting.get(node).getY() + offsets.getY(node) / norm * Math.min(norm, temperature))));
    }
    
    private void attraction(Edge edge) {
    	Point delta = layouting.get(edge.getFirst()).delta(layouting.get(edge.getSecond()));
    	double norm = Math.max(EPSILON, delta.norm());

    	double attraction = ATTRACTION / Math.sqrt(graph.getN());
    	double force = (norm * norm) / attraction;

    	double dx = force * (delta.getX() / norm);
    	double dy = force * (delta.getY() / norm);

    	offsets.get(edge.getFirst()).offset(-dx, -dy);
    	offsets.get(edge.getSecond()).offset(dx, dy);
    }
    
    private List<Integer> nodes;
    
    private void repulsion(int node) {
    	/* for (Integer other: graph.getNodes())
    		if (node != other && Math.random() < STOCHASTIC) {    			
    			Point delta = layouting.get(node).delta(layouting.get(other));
    			double norm = Math.max(EPSILON, delta.norm());
    			
    			double repulsion = REPULSION / Math.sqrt(graph.getN());
    			double force = (repulsion * repulsion) / norm;

    			offsets.get(node).offset(force * (delta.getX() / norm),  force * (delta.getY() / norm));
    		} */
    	
    	for (int i = 0; i < STOCHASTIC * graph.getN(); i++) {
    		int other = nodes.get((int)(Math.random() * nodes.size()));
    		if (node != other) {
    			Point delta = layouting.get(node).delta(layouting.get(other));
    			double norm = Math.max(EPSILON, delta.norm());

    			double repulsion = REPULSION / Math.sqrt(graph.getN());
    			double force = (repulsion * repulsion) / norm;

    			offsets.get(node).offset(force * (delta.getX() / norm),  force * (delta.getY() / norm));
    		}			
    	}
    }
    
}