package si.lj.uni.fmf.pro2.lectures.pacmans;

/**
 * Class representing a board of {@link Cell}-s in the Cartesian coordinate plane.
 * 
 * The initial board contains the specified number of randomly positioned {@link Pacman}-s with two prime {@link Pacman}-s. Other {@link Cell}-s are random {@link Coin}-s with probability {@link Pacmans#COINS_PROBABILITY}, {@link Block} cells with probability {@link Pacmans#BLOCKS_PROBABILITY} and {@link Empty} cells otherwise.
 * 
 * @author Lovro Šubelj
 */
public class Board {
	
	/**
	 * Board of {@link Cell}-s in the Cartesian coordinate plane.
	 */
	private Cell[][] board;
	
	/**
	 * Constructs a square board of {@link Cell}-s in the Cartesian coordinate plane with the specified size.
	 * 
	 * The board contains the specified number of randomly positioned {@link Pacman}-s with two prime {@link Pacman}-s. Other {@link Cell}-s are random {@link Coin}-s with probability {@link Pacmans#COINS_PROBABILITY} returned by {@link Coin#random(Position)}, {@link Block} cells with probability {@link Pacmans#BLOCKS_PROBABILITY} and {@link Empty} cells otherwise.
	 * 
	 * @param size width and height of the board of {@link Cell}-s
	 * @param pacmans number of {@link Pacman}-s of the board
	 */
	public Board(int size, int pacmans) {
		this(size, size, pacmans);
	}
	
	/**
	 * Constructs a rectangular board of {@link Cell}-s in the Cartesian coordinate plane with the specified width and height.
	 * 
	 * The board contains the specified number of randomly positioned {@link Pacman}-s with two prime {@link Pacman}-s. Other {@link Cell}-s are random {@link Coin}-s with probability {@link Pacmans#COINS_PROBABILITY} returned by {@link Coin#random(Position)}, {@link Block} cells with probability {@link Pacmans#BLOCKS_PROBABILITY} and {@link Empty} cells otherwise.
	 * 
	 * @param width width of the board of {@link Cell}-s
	 * @param height height of the board of {@link Cell}-s
	 * @param pacmans number of {@link Pacman}-s of the board
	 */
	public Board(int width, int height, int pacmans) {
		if (width <= 0 || height <= 0 || pacmans <= 0)
			throw new IllegalArgumentException();
		
		board = new Cell[width][height];
		
		for (int i = 0; i < pacmans; i++) {
			Position position = Position.random(this);
			board[position.getX()][position.getY()] = new Pacman(position, (char)(i + 65), i <= 1);
		}
		
		for (int i = 0; i < width; i++) 
			for (int j = 0; j < height; j++) 
				if (board[i][j] == null)
					board[i][j] = Cell.random(new Position(i, j));
	}
	
	/**
	 * Returns the width of the board of {@link Cell}-s in the Cartesian coordinate plane.
	 * 
	 * @return width of the board of {@link Cell}-s
	 */
	public int getWidth() {
		return board.length;
	}
	
	/**
	 * Returns the height of the board of {@link Cell}-s in the Cartesian coordinate plane.
	 * 
	 * @return height of the board of {@link Cell}-s
	 */
	public int getHeight() {
		return board[0].length;
	}

	/**
	 * Returns the {@link Cell} on the board in the Cartesian coordinate plane at the specified {@link Position}.
	 * 
	 * @param position {@link Position} on the board in the Cartesian coordinate plane
	 * 
	 * @return {@link Cell} on the board at the specified {@link Position}
	 */
	public Cell getCell(Position position) {
		return board[position.getX()][position.getY()];
	}
	
	/**
	 * Sets the specified {@link Cell} to the specified {@link Position} on the board in the Cartesian coordinate plane.
	 * 
	 * @param position {@link Position} on the board in the Cartesian coordinate plane
	 * @param cell new {@link Cell} on the board at the specified {@link Position}
	 */
	public void setCell(Position position, Cell cell) {
		board[position.getX()][position.getY()] = cell;
	}
	
	/**
	 * Returns the number of {@link Cell}-s on the board in the Cartesian coordinate plane that are instances of the specified {@link Class}.
	 * 
	 * @param cell {@link Class} of the {@link Cell}-s on the board
	 * 
	 * @return number of {@link Cell}-s on the board of the specified {@link Class}
	 */
	public int getCount(Class<?> cell) {
		int count = 0;
		for (int i = 0; i < getWidth(); i++) 
			for (int j = 0; j < getHeight(); j++) 
				if (getCell(new Position(i, j)).getClass().equals(cell))
					count++;
		
		return count;
	}

	/**
	 * Returns the value of the specified {@link Position} on the board in the Cartesian coordinate plane. The function returns {@code -Double.MAX_VALUE} if the specified {@link Position} is not on the board. Next, the function returns the value of the {@link Cell} at the specified {@link Position} returned by {@link Cell#getValue()} if the value is positive or the evaluation is greedy, or a randomly selected value with probability {@link Pacmans#GREEDY_PROBABILITY}. Otherwise, the function returns the sum of values of {@link Cell}-s on the board discounted by their distance from the specified {@link Position}.
	 * 
	 * @param position {@link Position} on the board in the Cartesian coordinate plane
	 * @param greedy whether the evaluation of the specified {@link Position} on the board is greedy
	 * 
	 * @return value of the specified {@link Position} on the board due to the specified strategy
	 */
	public double getValue(Position position, boolean greedy) {
		if (position.getX() < 0 || position.getX() >= getWidth() || position.getY() < 0 || position.getY() >= getHeight() || !getCell(position).isOccupiable())
			return -Double.MAX_VALUE;
		
		double value = getCell(position).getValue();
		if (greedy || value > 0)
			return value;
		
		if (Math.random() < Pacmans.GREEDY_PROBABILITY)
			return Math.random();
		
		value = 0.0;
		for (int i = 0; i < getWidth(); i++) 
			for (int j = 0; j < getHeight(); j++) 
				if (position.getX() != i || position.getY() != j)
					value += getCell(new Position(i, j)).getValue() / position.distance(i, j);
		
		return value / getWidth() / getHeight();
	}
	
	/**
	 * Returns the value of the specified {@link Position} on the board in the Cartesian coordinate plane. The function returns {@code -Double.MAX_VALUE} if the specified {@link Position} is not on the board. Otherwise, the function returns the value of the {@link Cell} at the specified {@link Position} returned by {@link Cell#getValue()}.
	 * 
	 * @param position {@link Position} on the board in the Cartesian coordinate plane
	 * 
	 * @return value of the specified {@link Position} on the board
	 */
	public double getValue(Position position) {
		return getValue(position, true);
	}

	/**
	 * Returns the sum of values of {@link Cell}-s on the board in the Cartesian coordinate plane returned by {@link Cell#getValue()}.
	 * 
	 * @return sum of values of {@link Cell}-s on the board
	 */
	public int getValue() {
		int value = 0;
		for (int i = 0; i < getWidth(); i++) 
			for (int j = 0; j < getHeight(); j++) 
				value += getCell(new Position(i, j)).getValue();
		
		return value;
	}

	/**
	 * Returns whether any occupiable {@link Cell} on the board in the Cartesian coordinate plane has positive value returned by {@link Cell#getValue()}.
	 * 
	 * @return whether any occupiable {@link Cell} on the board has positive value
	 */
	public boolean hasValue() {
		for (int i = 0; i < getWidth(); i++) 
			for (int j = 0; j < getHeight(); j++)
				if (getCell(new Position(i, j)).isOccupiable() && getCell(new Position(i, j)).getValue() > 0)
					return true;
		
		return false;
	}
	
	/**
	 * Returns whether any {@link Cell} on the board in the Cartesian coordinate plane is not either {@link Block} or {@link Coin}.
	 * 
	 * @return whether any {@link Cell} on the board is not either {@link Block} or {@link Coin}
	 */
	public boolean isEmpty() {
		for (int i = 0; i < getWidth(); i++) 
			for (int j = 0; j < getHeight(); j++)
				if (getCell(new Position(i, j)) instanceof Block || getCell(new Position(i, j)) instanceof Coin)
					return false;
		
		return true;
	}
	
	/**
	 * Returns {@link String} representation of the board of {@link Cell}-s in the Cartesian coordinate plane.
	 * 
	 * @return {@link String} representation of the board of {@link Cell}-s
	 */
	@Override
	public String toString() {
		String string = "   ";
		for (int i = 0; i < 2 * getWidth() + 1; i++)
			string += "-";
		string += "\n";
		
		for (int j = 0; j < getHeight(); j++) {
			string += "   |";
			for (int i = 0; i < getWidth(); i++)
				string += getCell(new Position(i, j)) + "|";
			string += "\n";
		}
		
		string += "   ";
		for (int i = 0; i < 2 * getWidth() + 1; i++)
			string += "-";
		
		return string;
	}

}