package si.lj.uni.fmf.pro2.lectures.pacmans;

/**
 * Class representing a cell on a {@link Board} represented by the Cartesian coordinate plane.
 * 
 * The cell is represented by its {@link Position} in the plane, its intrinsic value and a predefined label.
 * 
 * @author Lovro Šubelj
 */
public class Cell {
	
	/**
	 * {@link Position} of the cell in the Cartesian coordinate plane.
	 */
	private Position position;
	
	/**
	 * Intrinsic value of the cell equal to non-negative integer.
	 */
	private int value;
	
	/**
	 * Predefined label of the cell equal to single character.
	 */
	private char label;
	
	/**
	 * Boolean representing whether the cell is occupiable (e.g. empty).
	 */
	private boolean occupiable;

	/**
	 * Constructs a cell on a {@link Board} represented by the Cartesian coordinate plane.
	 * 
	 * The cell is specified by its {@link Position} in the plane, its intrinsic non-negative value, a predefined label as a single character and whether the cell is occupiable.
	 * 
	 * @param position {@link Position} of the cell on a {@link Board}
	 * @param value intrinsic non-negative value of the cell
	 * @param label predefined label or character of the cell
	 * @param occupiable whether the cell is occupiable
	 */
	public Cell(Position position, int value, char label, boolean occupiable) {
		if (value < 0)
			throw new IllegalArgumentException();
		
		this.position = position;
		this.value = value;
		this.label = label;
		this.occupiable = occupiable;
	}

	/**
	 * Returns the {@link Position} of the cell on a {@link Board}.
	 * 
	 * @return position of the cell on a {@link Board}
	 */
	public Position getPosition() {
		return position;
	}
	
	/**
	 * Sets the {@link Position} of the cell on a {@link Board} to the specified position.
	 * 
	 * @param position new position of the cell on a {@link Board}
	 */
	public void setPosition(Position position) {
		this.position = position;
	}

	/**
	 * Returns the value of the cell on a {@link Board} equal to non-negative integer.
	 * 
	 * @return value of the cell on a {@link Board}
	 */
	public int getValue() {
		return value;
	}

	/**
	 * Returns the label of the cell on a {@link Board} equal to single character.
	 * 
	 * @return label of the cell on a {@link Board}
	 */
	public char getLabel() {
		return label;
	}

	/**
	 * Returns whether the cell on a {@link Board} is occupiable (e.g. empty).
	 * 
	 * @return whether the cell on a {@link Board} is occupiable
	 */
	public boolean isOccupiable() {
		return occupiable;
	}

	/**
	 * Returns {@link String} representation of the cell on a {@link Board} equal to its label.
	 * 
	 * @return {@link String} representation of the cell on a {@link Board}
	 */
	@Override
	public String toString() {
		return "" + getLabel();
	}

	/**
	 * Returns randomly selected cell at the specified {@link Position} on a {@link Board}.
	 * 
	 * The function returns random {@link Coin} with probability {@link Pacmans#COINS_PROBABILITY}, {@link Block} cell with probability {@link Pacmans#BLOCKS_PROBABILITY} and {@link Empty} cell otherwise.
	 * 
	 * @param position {@link Position} of the cell on a {@link Board}
	 * 
	 * @return random cell at the specified {@link Position} on a {@link Board} 
	 */
	public static Cell random(Position position) {
		double random = Math.random();
		if (random < Pacmans.COINS_PROBABILITY)
			return Coin.random(position);
		else if (random < Pacmans.COINS_PROBABILITY + Pacmans.BLOCKS_PROBABILITY)
			return new Block(position);
		else
			return new Empty(position);
	}
	
}